import React from 'react';
import Layout from '../../components/shared/Layout';
import Sidebar from '../../components/shared/Sidebar';
import StatsCard from '../../components/shared/StatsCard';
import { deliverySidebarItems } from './DeliveryLayout';
import { 
  DollarSign, 
  Truck,
  MapPin,
  Clock,
  CheckCircle
} from 'lucide-react';

const DeliveryBoyDashboard: React.FC = () => {
  const stats = [
    {
      title: 'Total Earnings',
      value: '$3,450',
      icon: DollarSign,
      trend: { value: 15, isPositive: true },
      color: 'green' as const
    },
    {
      title: 'Deliveries Completed',
      value: '87',
      icon: Truck,
      trend: { value: 12, isPositive: true },
      color: 'blue' as const
    },
    {
      title: 'Pending Deliveries',
      value: '15',
      icon: Clock,
      trend: { value: 3, isPositive: false },
      color: 'orange' as const
    },
    {
      title: 'Total Distance',
      value: '245 km',
      icon: MapPin,
      trend: { value: 20, isPositive: true },
      color: 'purple' as const
    }
  ];

  const recentDeliveries = [
    { id: '#DEL-101', customer: 'Alice Johnson', amount: '$89.50', status: 'completed', time: '2 hours ago' },
    { id: '#DEL-102', customer: 'Bob Smith', amount: '$156.25', status: 'in_progress', time: '3 hours ago' },
    { id: '#DEL-103', customer: 'Carol Williams', amount: '$67.80', status: 'pending', time: '5 hours ago' },
    { id: '#DEL-104', customer: 'David Brown', amount: '$124.00', status: 'out_for_delivery', time: '6 hours ago' },
  ];

  const performanceMetrics = [
    { metric: 'On-Time Delivery', value: '92%', category: 'Performance' },
    { metric: 'Customer Satisfaction', value: '4.7/5', category: 'Rating' },
    { metric: 'Average Delivery Time', value: '45 min', category: 'Efficiency' },
    { metric: 'Packages Handled', value: '203', category: 'Volume' }
  ];

  const getStatusBadge = (status: string) => {
    const statusColors = {
      completed: 'bg-green-100 text-green-800',
      out_for_delivery: 'bg-blue-100 text-blue-800',
      in_progress: 'bg-yellow-100 text-yellow-800',
      pending: 'bg-gray-100 text-gray-800'
    };
    
    return `px-2 py-1 text-xs font-medium rounded-full ${statusColors[status as keyof typeof statusColors] || statusColors.pending}`;
  };

  return (
    <Layout title="Delivery Boy Dashboard" sidebar={<Sidebar items={deliverySidebarItems} />}>
      <div className="space-y-6">
        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {stats.map((stat, index) => (
            <StatsCard
              key={index}
              title={stat.title}
              value={stat.value}
              icon={stat.icon}
              trend={stat.trend}
              color={stat.color}
            />
          ))}
        </div>

        {/* Main Content Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Recent Deliveries */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Recent Deliveries</h3>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Delivery
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Customer
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Time
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {recentDeliveries.map((delivery) => (
                    <tr key={delivery.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">{delivery.id}</div>
                          <div className="text-sm text-gray-500">{delivery.amount}</div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {delivery.customer}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={getStatusBadge(delivery.status)}>
                          {delivery.status.replace('_', ' ')}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {delivery.time}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          {/* Performance Metrics */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Performance Metrics</h3>
            </div>
            <div className="p-6">
              <div className="space-y-4">
                {performanceMetrics.map((metric, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-blue-50 border border-blue-200 rounded-lg">
                    <div className="flex-1">
                      <p className="text-sm font-medium text-gray-900">{metric.metric}</p>
                      <p className="text-xs text-gray-500">{metric.category}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-medium text-blue-600">
                        {metric.value}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
              <button className="w-full mt-4 px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors">
                View Full Performance Report
              </button>
            </div>
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-white rounded-lg shadow-sm border">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-medium text-gray-900">Quick Actions</h3>
          </div>
          <div className="p-6">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              {[
                { label: 'Start Delivery', icon: Truck, count: '15 pending' },
                { label: 'Update Status', icon: CheckCircle, count: '8 in progress' },
                { label: 'View Route', icon: MapPin, count: '3 routes' },
                { label: 'Collect Payment', icon: DollarSign, count: '6 due' }
              ].map((action, index) => {
                const Icon = action.icon;
                return (
                  <button
                    key={index}
                    className="p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-all text-left"
                  >
                    <div className="flex items-center mb-2">
                      <Icon className="w-5 h-5 text-blue-600 mr-2" />
                      <span className="text-sm font-medium text-gray-900">{action.label}</span>
                    </div>
                    <p className="text-xs text-gray-500">{action.count}</p>
                  </button>
                );
              })}
            </div>
          </div>
        </div>
      </div>
    </Layout>
  );
};

export default DeliveryBoyDashboard;
